import { registerAppResource, registerAppTool, RESOURCE_MIME_TYPE } from "@modelcontextprotocol/ext-apps/server";
import { McpServer } from "@modelcontextprotocol/sdk/server/mcp.js";
import fs from "node:fs/promises";
import path from "node:path";
import { z } from "zod";

//JAXA Earth APIを使った個別の機能
import getElevationImage from "./getElevationImage.js";

export default (): McpServer => {

	//MCPサーバーを作成
	const server = new McpServer({
		name: "jaxa-earth-api",
		version: "1.0.0",
	});

	//UIリソースのURIを定義
	const resourceUri = "ui://get-elevation-image/ui";

	//ツールを定義
	registerAppTool(
		server,
		"jaxa-earth-api-get-elevation-image",
		{
			title: "地形データ表示",
			description: `
				JAXA Earth APIを使って全世界の標高のデータを画像で取得します。
				標高の値を知りたい場所の経度(longitude)と緯度(latitude)を指定すると、その点を中心とした緯度経度で半径1度の範囲の標高データを画像にして返します。
				500ピクセル四方のPNG画像として返します。
				画像の色は、標高0m～6000mを青→水色→緑→黄色→赤で塗りつぶしたものです。
				海上など、観測データが無い部分は透明になります。
				源泉となるデータはAW3Dです。データセットの詳細について知りたい場合は、
				https://data.earth.jaxa.jp/ja/datasets/#/id/JAXA.EORC_ALOS.PRISM_AW3D30.v3.2_global
				をご参照ください。
			`,
			inputSchema: {
				longitude: z.number().describe("経度を指定します。（例：135.5）"),
				latitude: z.number().describe("緯度を指定します。（例：35.8）"),
			},
			_meta: { ui: { resourceUri } },
		},
		async ({ longitude, latitude }) => {
			console.log(longitude, latitude);

			return {
				content: [{
					type: "image",
					//バイナリデータはBASE64にして返します。
					data: Buffer.from(await getElevationImage(longitude, latitude)).toString("base64"),
					mimeType: "image/png",
				}],
			};
		},
	);

	//リソースを定義
	registerAppResource(
		server,
		resourceUri,
		resourceUri,
		{ mimeType: RESOURCE_MIME_TYPE },
		async () => {
			
			//このcreateMcpServer.tsのコンパイル出力先フォルダbuild（=import.meta.dirname）内に、
			//コンパイル後のindex.htmlもある場合
			const text = await fs.readFile(path.join(import.meta.dirname, "index.html"), "utf-8");

			return {
				contents: [{
					uri: resourceUri,
					mimeType: RESOURCE_MIME_TYPE,
					text,
				}],
			};
		},
	);
	
	return server;
}