
import { McpServer } from "@modelcontextprotocol/sdk/server/mcp.js";
import { z } from "zod";

//JAXA Earth APIを使った各機能
import getCatalogList from "./getCatalogList.js";
import getLinks from "./getLinks.js";
import getValue from "./getValue.js";
import getImage from "./getImage.js";
import getCsv from "./getCsv.js";
import getJavaScriptCode from "./getJavaScriptCode.js";
import getDocs from "./getDocs.js";

export default (): McpServer => {

	//MCPサーバーを作成
	const server = new McpServer({
		name: "jaxa-earth-api",
		version: "1.0.0",
	});

	//----------------------------------------------------------------------------------------------------

	server.registerTool(
		"get-catalog-list",
		{
			description: `
				JAXA Earth APIを使って利用できる全てのデータセットの情報をマークダウンのリストとして取得します。
				リストの項目の各詳細についてはマークダウンの中に書かれています。
			`,
			inputSchema: {},
		},
		async ({ }) => {
			return {
				content: [
					{
						type: "text",
						text: String(await getCatalogList()),
					},
				],
			};
		},
	);

	//----------------------------------------------------------------------------------------------------

	server.registerTool(
		"get-links",
		{
			description: `
				データセットの詳細に関するページのリンクを生成します。
				各データセットの詳細を確認できるページや、簡易的な地図表示が可能なページのリンクを得られます。
				回答作成の際にユーザーにとって有用なページへのリンクを付加できますので、積極的に回答に含めてください。
				利用できるデータセットのcollection.jsonのURLとband、選択できるdateの範囲については、get-catalog-listのツールを実行すると得られます。
			`,
			inputSchema: {
				collectionUrl: z.string().describe("利用したいデータセットのcollection.jsonのURLを指定します。"),
				date: z.string().describe("日時を指定します。ISO8601形式の文字列として指定します。（例：2025-11-01T00:00:00Z）"),
				band: z.string().describe("利用したいデータセットのbandを指定します。"),
			},
		},
		async ({ collectionUrl, date, band }) => {
			return {
				content: [
					{
						type: "text",
						text: String(getLinks(collectionUrl, new Date(date), band)),
					},
				],
			};
		},
	);

	//----------------------------------------------------------------------------------------------------

	server.registerTool(
		"get-value",
		{
			description: `
				JAXA Earth APIを使ってデータを数値で取得します。
				値を知りたい日時(date)、場所の経度(longitude)と緯度(latitude)を指定すると、その場所（1点のみ）の値を返します。
				観測データが無い場合はNaNまたはエラーを返します。
				利用できるデータセットのcollection.jsonのURLとband、選択できるdateの範囲については、get-catalog-listのツールを実行すると得られます。
				一度に50点四方、計2500点の値を取得したい場合は、get-csvのツールを利用してください。
			`,
			inputSchema: {
				collectionUrl: z.string().describe("利用したいデータセットのcollection.jsonのURLを指定します。"),
				date: z.string().describe("日時を指定します。ISO8601形式の文字列として指定します。（例：2025-11-01T00:00:00Z）"),	//文字列として指定しないとセッションタイムアウトする
				band: z.string().describe("利用したいデータセットのbandを指定します。"),
				longitude: z.number().describe("経度を指定します。（例：135.5）"),
				latitude: z.number().describe("緯度を指定します。（例：35.8）"),
			},
		},
		async ({ collectionUrl, date, band, longitude, latitude }) => {

			const res = await getValue(collectionUrl, new Date(date), band, longitude, latitude);

			return {
				content: [
					{
						type: "text",
						text: String(res.value + res.unit),
					},
				],
			};
		},
	);

	//----------------------------------------------------------------------------------------------------

	server.registerTool(
		"get-image",
		{
			description: `
				JAXA Earth APIを使ってデータを画像で取得します。統計値も同時に取得します。
				値を知りたい日時(date)、場所の経度(longitude)と緯度(latitude)を指定すると、その点を中心とした緯度経度で半径dl度の範囲のデータを画像にして返します。
				1000ピクセル四方のPNG画像として返します。
				画像の色は、最小値～最大値を青→水色→緑→黄色→赤で塗りつぶしたものです。
				観測データが無い部分は透明になります。
				緯度経度範囲、最小値、最大値、平均値、値の単位も同時に返します。
				
				利用できるデータセットのcollection.jsonのURLとband、選択できるdateの範囲については、get-catalog-listのツールを実行すると得られます。
			`,
			inputSchema: {
				collectionUrl: z.string().describe("利用したいデータセットのcollection.jsonのURLを指定します。"),
				date: z.string().describe("日時を指定します。ISO8601形式の文字列として指定します。（例：2025-11-01T00:00:00Z）"),
				band: z.string().describe("利用したいデータセットのbandを指定します。"),
				longitude: z.number().describe("経度を指定します。（例：135.5）"),
				latitude: z.number().describe("緯度を指定します。（例：35.8）"),
				dl: z.number().describe("緯度経度の半径を指定します。（例：1.0とすると、指定した緯度経度から±1.0度の範囲でデータを取得します）"),
			},
		},
		async ({ collectionUrl, date, band, longitude, latitude, dl }) => {

			const res = await getImage(collectionUrl, new Date(date), band, longitude, latitude, dl);

			return {
				content: [
					{
						type: "image",
						data: Buffer.from(res.image).toString("base64"),
						mimeType: "image/png",
					},
					{
						type: "text",
						text: String([
							"経度範囲: " + res.bbox[0] + "～" + res.bbox[2],
							"緯度範囲: " + res.bbox[1] + "～" + res.bbox[3],
							"最小値: " + res.min,
							"最大値: " + res.max,
							"平均値: " + res.mean,
							"単位: " + res.unit,
						].join(", ")),
					},
				],
			};
		},
	);

	//----------------------------------------------------------------------------------------------------

	server.registerTool(
		"get-csv",
		{
			description: `
				JAXA Earth APIを使ってデータをCSVとして取得します。統計値も同時に取得します。
				値を知りたい日時(date)、場所の経度(longitude)と緯度(latitude)を指定すると、その点を中心とした緯度経度で半径dl度の範囲のデータをCSVにして返します。
				50ピクセル四方のデータとして取得し、50 * 50 = 計2500組の経度、緯度、値の組をCSVとして返します。
				観測データが無い点についてはNaNになります。
				緯度経度範囲、最小値、最大値、平均値、値の単位も同時に返します。
				指定された範囲内で、指定された値の場所（緯度経度）を探すような場合に利用してください。
				
				利用できるデータセットのcollection.jsonのURLとband、選択できるdateの範囲については、get-catalog-listのツールを実行すると得られます。
				分解能がさらに高い1000ピクセル四方のPNG画像としてデータを得たい場合はget-imageのツールも併用してください。
			`,
			inputSchema: {
				collectionUrl: z.string().describe("利用したいデータセットのcollection.jsonのURLを指定します。"),
				date: z.string().describe("日時を指定します。ISO8601形式の文字列として指定します。（例：2025-11-01T00:00:00Z）"),
				band: z.string().describe("利用したいデータセットのbandを指定します。"),
				longitude: z.number().describe("経度を指定します。（例：135.5）"),
				latitude: z.number().describe("緯度を指定します。（例：35.8）"),
				dl: z.number().describe("緯度経度の半径を指定します。（例：1.0とすると、指定した緯度経度から±1.0度の範囲でデータを取得します）"),
			},
		},
		async ({ collectionUrl, date, band, longitude, latitude, dl }) => {

			const res = await getCsv(collectionUrl, new Date(date), band, longitude, latitude, dl);

			return {
				content: [
					{
						type: "text",
						text: String([
							"経度範囲: " + res.bbox[0] + "～" + res.bbox[2],
							"緯度範囲: " + res.bbox[1] + "～" + res.bbox[3],
							"最小値: " + res.min,
							"最大値: " + res.max,
							"平均値: " + res.mean,
							"単位: " + res.unit,
						].join(", ") + "\n\nCSVデータ:\n" + res.data),
					},
				],
			};
		},
	);

	//----------------------------------------------------------------------------------------------------

	server.registerTool(
		"get-javascript-code",
		{
			description: `
				JAXA Earth APIを使ってデータを取得するためのJavaScriptコード（ブラウザ環境での実行用）を生成します。
				値を知りたい日時(date)、場所の経度(longitude)と緯度(latitude)を指定すると、その点を中心とした緯度経度で半径dl度の範囲のデータを返すためのコードを生成します。
				利用できるデータセットのcollection.jsonのURLとband、選択できるdateの範囲については、get-catalog-listのツールを実行すると得られます。
				APIドキュメントについてはget-api-docsのツールを実行すると得られます。
			`,
			inputSchema: {
				collectionUrl: z.string().describe("利用したいデータセットのcollection.jsonのURLを指定します。"),
				date: z.string().describe("日時を指定します。ISO8601形式の文字列として指定します。（例：2025-11-01T00:00:00Z）"),
				band: z.string().describe("利用したいデータセットのbandを指定します。"),
				longitude: z.number().describe("経度を指定します。（例：135.5）"),
				latitude: z.number().describe("緯度を指定します。（例：35.8）"),
				dl: z.number().describe("緯度経度の半径を指定します。（例：1.0とすると、指定した緯度経度から±1.0度の範囲でデータを取得します）"),
			},
		},
		async ({ collectionUrl, date, band, longitude, latitude, dl }) => {
			return {
				content: [
					{
						type: "text",
						text: String(getJavaScriptCode(collectionUrl, new Date(date), band, longitude, latitude, dl)),
					},
				],
			};
		},
	);

	//----------------------------------------------------------------------------------------------------

	server.registerTool(
		"get-api-docs",
		{
			description: `
				JAXA Earth API for JavaScriptを利用するための方法が書かれたAPIドキュメント全体をマークダウン形式で取得します。
				複雑なソースコードを作成する際に利用してください。
				簡単なソースコードについてはget-javascript-codeのツールを実行しても得られます。
			`,
			inputSchema: {},
		},
		async ({ }) => {
			return {
				content: [
					{
						type: "text",
						text: String(await getDocs()),
					},
				],
			};
		},
	);
	
	//----------------------------------------------------------------------------------------------------

	return server;
}