
import { McpServer } from "@modelcontextprotocol/sdk/server/mcp.js";
import { z } from "zod";

//JAXA Earth APIを使った各機能
import getElevationValue from "./getElevationValue.js";
import getElevationImage from "./getElevationImage.js";
import getCatalogList from "./getCatalogList.js";

export default (): McpServer => {

	//MCPサーバーを作成
	const server = new McpServer({
		name: "jaxa-earth-api",
		version: "1.0.0",
	});

	//利用できるツールを登録
	server.registerTool(
		"jaxa-earth-api-get-elevation-value",
		{
			//このツールの説明を記載
			description: `
				JAXA Earth APIを使って全世界の標高のデータを数値で取得します。
				標高の値を知りたい場所の経度(longitude)と緯度(latitude)を指定すると、その場所の標高(単位はm)を返します。
				海上など、観測データが無い場合はNaNまたはエラーを返します。
				源泉となるデータはAW3Dです。データセットの詳細について知りたい場合は、
				https://data.earth.jaxa.jp/ja/datasets/#/id/JAXA.EORC_ALOS.PRISM_AW3D30.v3.2_global
				をご参照ください。
			`,
			//受け付ける引数を登録
			inputSchema: {
				longitude: z.number().describe("経度を指定します。（例：135.5）"),
				latitude: z.number().describe("緯度を指定します。（例：35.8）"),
			},
		},
		//引数を受け付けて実行する関数
		async ({ longitude, latitude }) => {
			return {
				content: [
					{
						type: "text",
						//結果の数値については文字列として返します。
						text: String(await getElevationValue(longitude, latitude)),
					},
				],
			};
		},
	);

	//----------------------------------------------------------------------------------------------------

	server.registerTool(
		"jaxa-earth-api-get-elevation-image",
		{
			description: `
				JAXA Earth APIを使って全世界の標高のデータを画像で取得します。
				標高の値を知りたい場所の経度(longitude)と緯度(latitude)を指定すると、その点を中心とした緯度経度で半径1度の範囲の標高データを画像にして返します。
				500ピクセル四方のPNG画像として返します。
				画像の色は、標高0m～6000mを青→水色→緑→黄色→赤で塗りつぶしたものです。
				海上など、観測データが無い部分は透明になります。
				源泉となるデータはAW3Dです。データセットの詳細について知りたい場合は、
				https://data.earth.jaxa.jp/ja/datasets/#/id/JAXA.EORC_ALOS.PRISM_AW3D30.v3.2_global
				をご参照ください。
			`,
			inputSchema: {
				longitude: z.number().describe("経度を指定します。（例：135.5）"),
				latitude: z.number().describe("緯度を指定します。（例：35.8）"),
			},
		},
		async ({ longitude, latitude }) => {
			return {
				content: [
					{
						type: "image",
						//バイナリデータはBASE64にして返します。
						data: Buffer.from(await getElevationImage(longitude, latitude)).toString("base64"),
						mimeType: "image/png",
					},
				],
			};
		},
	);

	//----------------------------------------------------------------------------------------------------

	server.registerTool(
		"jaxa-earth-api-get-catalog-list",
		{
			description: `
				JAXA Earth APIを使って利用できるデータを案内します。
			`,
			inputSchema: {

			},
		},
		async ({ }) => {
			return {
				content: [
					{
						type: "text",
						text: String(await getCatalogList()),
					},
				],
			};
		},
	);
	
	return server;
}