# Import
import datetime
from pathlib import Path
from osgeo import gdal
from .meta.get import geo as get_geo_meta

# set_params_tags : 
def params_tags(img, band, ctable, cog, idx):

    # Set general options
    # PhotometricInterpretation
    # - 0 (white is zero), MINISBLACK
    # - 1 (black is zero), MINISWHITE
    # - 2 (RGB)          , RGB
    # - 3 (Palette Color), PALETTE
    gopt = ["TILED=YES",
           f"PHOTOMETRIC={cog['bands'][band]['pint']}",
            "BLOCKXSIZE=256",
            "BLOCKYSIZE=256",
            "PREDICTOR=1",
            "COMPRESS=DEFLATE"]    

    # Set img,lat,lon parameters
    img_size = img.shape
    latlim   = cog["bbox_all"][idx][1::2]
    lonlim   = cog["bbox_all"][idx][0::2]
    dlat     = (latlim[1]-latlim[0])/img_size[1]
    dlon     = (lonlim[1]-lonlim[0])/img_size[2]

    # Set longitude folder and tif name
    fn_dir, fn_tif = fname(cog,idx,band)

    # Create longitude folder (if not exist)
    path_tmp = Path(cog["out_path"]).joinpath(fn_dir)
    if not path_tmp.exists():
        path_tmp.mkdir(parents=True)

    # Set tiff full path
    tif_full_path = path_tmp.joinpath(fn_tif)
    
    # Set parameters dict
    params = {
        "img": img,
        "img_size": img_size,
        "dtype": gdal_dtype(img.dtype),
        "epsg": cog["epsg"],
        "latlim": latlim,
        "lonlim": lonlim,
        "dlat": dlat,
        "dlon": dlon,
        "fn_out": tif_full_path,
        "img_lv": cog["overviews"],
    }

    # Set Tag and metadata Information
    custom_tags = {
        "ColorMap": ctable,
        "GDAL_NODATA": cog["bands"][band]["dn"]["nodata"],
        "GDAL_METADATA":{
            "unit"  : cog["bands"][band]["value"]["unit"] ,
            "scale" : cog["bands"][band]["dn2value"]["slope"],
            "offset": cog["bands"][band]["dn2value"]["offset"],
        },
    }

    # Set metadata tag
    this_year = datetime.datetime.now().year
    EORC      = "Earth Observation Research Center (EORC)"
    JAXA      = "Japan Aerospace Exploration Agency (JAXA)"
    copyright = f'{this_year} {EORC}, {JAXA}'
    software  = 'JAXA Earth Tool for Python: Data Generator'
    metadata  = {
        'AREA_OR_POINT':'AREA',
        'GEO_METADATA': get_geo_meta(cog["dlim"]),
        'TIFFTAG_COPYRIGHT': copyright,
        'TIFFTAG_DOCUMENTNAME': fn_tif,
        'TIFFTAG_DATETIME': cog["dlim"][0],
        'TIFFTAG_SOFTWARE':software,
    }

    # Return
    return gopt, params, custom_tags, metadata

# Set file name of cog
def fname(cog,idx,band):

    # Set bbox
    bbox = cog["bbox_all"][idx]

    # Set x
    x     = bbox[0::2]
    x_str = []
    for i in range(len(x)):

        # Detect plus/minus
        if x[i] >= 0:
            x_str_tmp = cog["x_plus_str"]
        else:
            x_str_tmp = cog["x_minus_str"]

        # Append
        x_str.append(x_str_tmp+cog["x_strfmt"].format(abs(x[i])))

    # Set y
    y     = bbox[1::2]
    y_str = []
    for i in range(len(y)):

        # Detect plus/minus
        if y[i] >= 0:
            y_str_tmp = cog["y_plus_str"]
        else:
            y_str_tmp = cog["y_minus_str"]

        # Append
        y_str.append(y_str_tmp+cog["y_strfmt"].format(abs(y[i])))

    # Merge
    fn_dir = f"{x_str[0]}-{x_str[1]}"
    fn_tif = f"{x_str[0]}-{y_str[0]}-{x_str[1]}-{y_str[1]}-{band}.tiff"

    # Return
    return fn_dir, fn_tif

# Set gdal data type
def gdal_dtype(dtype):

    # Check data type of images
    if   dtype ==   "int16": g_dtype = gdal.GDT_Int16
    elif dtype ==   "int32": g_dtype = gdal.GDT_Int32
    elif dtype ==   "int64": g_dtype = gdal.GDT_Int64
    elif dtype ==  "uint8" : g_dtype = gdal.GDT_Byte
    elif dtype ==  "uint16": g_dtype = gdal.GDT_UInt16
    elif dtype ==  "uint32": g_dtype = gdal.GDT_UInt32
    elif dtype ==  "uint64": g_dtype = gdal.GDT_UInt64
    elif dtype == "float32": g_dtype = gdal.GDT_Float32
    elif dtype == "float64": g_dtype = gdal.GDT_Float64

    # Return
    return g_dtype