# Import
from pathlib import Path
from datetime import datetime,timedelta
import netCDF4 as nc4
import numpy as np
import math

# Read JASMES SGLI Standard data
def sgli_std(fn,output_path,vname):

    # Detect global or japan
    if "3MSG" in fn.stem:
        global_product = True
    else:
        global_product = False

    # Read file
    nc = nc4.Dataset(fn,"r")

    # Read lat,lon information
    grid_interval = nc.grid_interval
    bbox = np.array([ nc.lower_left_longitude  - grid_interval/2,
                      nc.lower_left_latitude   - grid_interval/2, 
                      nc.upper_right_longitude + grid_interval/2,
                      nc.upper_right_latitude  + grid_interval/2])
    if global_product:
        bbox = bbox + np.array([-180,0,-180,0])

    # Get start time
    #start_time = nc.variables["start_time"][:].data[0]
    #date_str   = get_nearest_date(start_time)

    # Get date str
    date_str = fn.stem[7:11]+"-"+fn.stem[11:13]+"-"+fn.stem[13:15]

    # Read variable, metadata
    data   = nc.variables[vname]
    unit   = data.units
    slope  = data.scale_factor
    offset = data.add_offset
    nodata = data.missing_value

    # Set auto mask and scale to False
    data.set_auto_maskandscale(False)

    # Get raw image as array
    if global_product:
        img_tmp = data[:]
        img = np.hstack([img_tmp[:,3600:],img_tmp[:,0:3600]])
    else:
        img = data[:]

    # Close
    nc.close()

    # Create file name
    fn_tiff = fn.stem+"-"+vname+".tiff"
    fn_out  = Path(output_path).joinpath(date_str)\
                               .joinpath(fn_tiff)

    # Set output
    output = {
        "fn_out": fn_out,
        "img" : img,
        "bbox": bbox,
        "ddeg": grid_interval,
        "unit": unit,
        "slope": slope,
        "offset": offset,
        "nodata": nodata
    }

    # Return
    return output

# Get nearest date
def get_nearest_date(ftime):

    # ftime to str
    ftime_str = "{:15.6f}".format(ftime)

    # str to datetime
    date = datetime.strptime(ftime_str,"%Y%m%d.%H%M%S")

    # Set time delta to round date
    add_day = timedelta(days = round(date.hour/24))

    # Add days
    date = date + add_day

    # Convert to str
    date_str = datetime.strftime(date,"%Y-%m-%d")
    
    # Return
    return date_str
