# Import
from pathlib import Path
from datetime import datetime,timedelta
import netCDF4 as nc4
import numpy as np
import math

# Read JASMES SGLI Standard data
def fulldisk(fn,output_path,vname):

    # Read file
    nc = nc4.Dataset(fn,"r")

    # Read lat,lon information
    ddeg_lat = nc.geospatial_lat_resolution
    ddeg_lon = nc.geospatial_lat_resolution
    bbox = [ 80 - ddeg_lon/2,
            -60 - ddeg_lat/2,
            200 + ddeg_lon/2,
             60 + ddeg_lat/2]

    # Get date str
    date_str = fn.stem[4:8]+"-"+fn.stem[8:10]#+"-"+fn.stem[10:12]

    # Read variable, metadata
    data   = nc.variables[vname]
    unit   = data.units
    slope  = data.scale_factor
    offset = data.add_offset
    nodata = data._FillValue

    # Set auto mask and scale to False
    data.set_auto_maskandscale(False)

    # Get raw image as array
    img = data[:]

    # Close
    nc.close()

    # Create file name
    fn_tiff = fn.stem+"-"+vname+".tiff"
    fn_out  = Path(output_path).joinpath(date_str)\
                               .joinpath(fn_tiff)

    # Set output
    output = {
        "fn_out": fn_out,
        "img" : img,
        "bbox": bbox,
        "ddeg": ddeg_lat,
        "unit": unit,
        "slope": slope,
        "offset": offset,
        "nodata": nodata
    }

    # Return
    return output

# Get nearest date
def get_nearest_date(ftime):

    # ftime to str
    ftime_str = "{:15.6f}".format(ftime)

    # str to datetime
    date = datetime.strptime(ftime_str,"%Y%m%d.%H%M%S")

    # Set time delta to round date
    add_day = timedelta(days = round(date.hour/24))

    # Add days
    date = date + add_day

    # Convert to str
    date_str = datetime.strftime(date,"%Y-%m-%d")
    
    # Return
    return date_str
