# Import
import pystac
from pathlib import Path
import rasterio
from shapely.geometry import Polygon, mapping
from ....date.set import set_dlim

# tiff
def tiff(collection,endpoint,product_dict,lon_folder,lon_str,lat_str):

    # Detect tiff
    tiff_str   = lon_str.split("-")[0] + "-" +\
                 lat_str.split("-")[0] + "-" +\
                 lon_str.split("-")[1] + "-" +\
                 lat_str.split("-")[1]
    tiff_names = list(Path(lon_folder).glob("*"+tiff_str+"*"))

    # Get bbox, footprint
    bbox, footprint = get_bbox_and_footprint(tiff_names[0])

    # Set params for dlim
    dfmt     = product_dict["summaries"]["je:stac_date_format"][0]
    duration = product_dict["duration"]
    ext_t    = product_dict["extent"]["temporal"]["interval"][0]

    # Detect date str and depth
    dfmt_snum = dfmt.count("/")
    if   dfmt_snum == 0:
        date_str = lon_folder.parts[-3]
    elif dfmt_snum == 1:
        date_str = "-".join(lon_folder.parts[-4:-2:])
    else:
        raise Exception("Error! 3 depth is not inpremented !")

    # Set dlim function
    dlim_str,dstr = set_dlim(dfmt,date_str,duration,ext_t)

    # Set properties
    prop = {
        "start_datetime": dlim_str[0],
        "end_datetime": dlim_str[1],
        "collection": product_dict["id"],
        "license": "proprietary",
        "je:cog_level": int(lon_folder.parts[-2]),
        "je:stac_version": endpoint["version_full"]
    }

    # Set extensions
    extensions = [
        "https://stac-extensions.github.io/classification/v1.1.0/schema.json"
    ]

    # Make tiff item
    item = pystac.Item(
        id         = lat_str,
        collection = product_dict["id"],
        geometry   = footprint,
        bbox       = bbox,
        datetime   = None,
        properties = prop
    )

    # Detect band name and add asset
    for i in range(len(tiff_names)):

        # Detect band name from tiff
        band = tiff_names[i].parts[-1].split("-")[-1].split(".")[0]
  
        # Set item's href (relative)
        path_on_cloud = "./"+tiff_names[i].parts[-1]

        # Extra fields
        extra_fields = {
            "je:rasters":{
                "value": product_dict["assets"][band]["cog"]["value"],
                "dn2value": product_dict["assets"][band]["cog"]["dn2value"],
                "dn": product_dict["assets"][band]["cog"]["dn"],
            }
        }

        # Set classes if exists
        labels = product_dict["assets"][band]["classification:classes"]
        if not (not labels):

            # Set classification extension
            item.stac_extensions = extensions

            # Set classification fields
            extra_fields["classification:classes"] = labels

        # Add asset
        item.add_asset(
            key   = band,
            asset = pystac.Asset(
                title = product_dict["assets"][band]["title"],
                roles = product_dict["assets"][band]["roles"],
                href  = str(path_on_cloud),
                media_type   = pystac.MediaType.COG,
                extra_fields = extra_fields
            )
        )

        # Set owner to execute relative link
        #item.assets[band].set_owner("Collection")

    # license link
    link_licence = pystac.Link(
        rel        = pystac.RelType.LICENSE,
        target     = product_dict["links"][0]["href"],
        media_type = "text/html"
    )

    # Collection link
    link_collection = pystac.Link(
        rel        = pystac.RelType.COLLECTION,
        target     = collection,
        media_type = pystac.media_type.MediaType.JSON,
    )

    # Add link (can't add collection's link)
    item.add_links([link_licence,link_collection])

    # Save object
    #item.save_object(dest_href="G://test")
    
    # Return name
    return item

# Get bbox and footprint
def get_bbox_and_footprint(raster_path):

    # Get information from raster
    with rasterio.open(raster_path) as ds:

        # Bounds
        bounds = ds.bounds
        bbox   = [bounds.left, 
                  bounds.bottom, 
                  bounds.right, 
                  bounds.top]

        # Geometry
        footprint = Polygon([
            [bounds.left , bounds.bottom],
            [bounds.left , bounds.top   ],
            [bounds.right, bounds.top   ],
            [bounds.right, bounds.bottom]
        ])

        # Return
        return (bbox, mapping(footprint))
