# Import
import pystac
from datetime import datetime
from pystac.extensions import scientific
from .catalog.gen import date_of_one_depth as gen_date_catalog1
from .catalog.gen import date_of_two_depth as gen_date_catalog2
from pathlib import Path

# Generate collection stac
def collection(endpoint,product_dict):

    # Display status
    print(f" - Collection ID : {product_dict['id']}")

    # Make collection
    collection = collection_core(endpoint,product_dict)

    # Get collection's path
    root_id = Path(endpoint["root_local"])\
                   .joinpath(endpoint["bucket"])\
                   .joinpath(endpoint["ftype"])\
                   .joinpath(endpoint["version_path"])\
                   .joinpath(product_dict["id"])

    # Get all child (date folders)
    date_folders = [p.parts[-1] for p in root_id.iterdir() if p.is_dir()]

    # Get date format
    dfmt = product_dict["summaries"]["je:stac_date_format"][0]
    dfmt_snum = dfmt.count("/")

    # Make and add child
    for i in range(len(date_folders)):

        # Detect folder, and make
        date_folder  = root_id.joinpath(date_folders[i])

        # Make child (depth = 1)
        if   dfmt_snum == 0: 
            date_catalog = gen_date_catalog1(collection,endpoint,product_dict,date_folder,date_folders[i])

        # Make child (depth = 2)
        elif dfmt_snum == 1:
            date_catalog = gen_date_catalog2(collection,endpoint,product_dict,date_folder,date_folders[i])

        # Make child (depth = 3)
        elif dfmt_snum == 2:
            raise Exception("Error! 3 depth is not inpremented !")

        # Make child (depth = ?)
        else:
            raise Exception("Error! Over 3 depth is not defined !")

        # Add child
        collection.add_child(date_catalog)

    # Display progress
    print(" - STAC saving ... ",end="")

    # Avoid making another folder in catalog
    strategy = pystac.layout.TemplateLayoutStrategy(
        item_template = "",
    )

    # Set STAC root path
    stac_tmp_folder = str(root_id.as_posix())+"/"

    # Normalize and save
    # ABSOLUTE_PUBLISHED
    # RELATIVE_PUBLISHED
    # SELF_CONTAINED
    collection.normalize_and_save(
        root_href    = stac_tmp_folder,
        catalog_type = pystac.CatalogType.SELF_CONTAINED,
        strategy     = strategy
    )
    print("done")

    # Validate collection
    #collection.validate_all()

    # Return
    return 1

# Make collection's core
def collection_core(endpoint,product_dict):

    # Set spatial extent
    sp_extent  = pystac.SpatialExtent([product_dict["extent"]["spatial"]["bbox"][0]])

    # Set temporal extent
    dlim = []
    dfmt = "%Y-%m-%dT%H:%M:%SZ"
    for i in range(2):
        time_tmp = product_dict["extent"]["temporal"]["interval"][0][i]
        if time_tmp is not None:
            dlim.append(datetime.strptime(time_tmp,dfmt))
        else:
            dlim.append(None)
    tmp_extent = pystac.TemporalExtent([dlim])

    # Set extent
    extent = pystac.Extent(sp_extent,tmp_extent)

    # Set providers
    providers_raw = product_dict["providers"]
    providers = []
    for i in range(len(providers_raw)):
        providers.append(pystac.Provider.from_dict(providers_raw[i]))

    # Set summaries
    #summaries = pystac.Summaries.empty()
    #summaries.add("platform",["Global Change Observation Mission - Climate (GCOM-C)"])
    summaries = pystac.Summaries(product_dict["summaries"])

    # Set extensions
    extensions = [
        scientific.ScientificExtensionHooks.schema_uri
    ]

    # Collection stac
    collection = pystac.Collection(
        stac_extensions = extensions,
        id          = product_dict["id"],
        providers   = providers,
        title       = product_dict["title"],
        description = product_dict["description"],
        keywords    = product_dict["keywords"],
        extent      = extent,
        license     = "proprietary",
        summaries   = summaries,
    )

    # Add asset
    bands = list(product_dict["assets"].keys())
    for i in range(len(bands)):

        # Set asset
        asset = pystac.Asset(
            title = product_dict["assets"][bands[i]]["title"],
            media_type = pystac.MediaType.COG,
            href  = "N/A",#asset_root,
            roles = product_dict["assets"][bands[i]]["roles"]
        )

        # Add asset
        collection.add_asset(bands[i],asset)

    # Set links
    link_licence = pystac.Link(
        rel        = pystac.RelType.LICENSE,
        target     = product_dict["links"][0]["href"],
        media_type = "text/html"
    )

    # Add links
    collection.add_links([link_licence])

    # Set scientific extension to collection
    sci_ext  = scientific.ScientificExtension.ext(collection)
    sci_dict = product_dict["sci:publications"]
    pub = []
    for i in range(len(sci_dict)):

        # Check doi
        if "doi" in sci_dict[i]:
            doi = sci_dict[i]["doi"]
        else:
            doi = None

        # Make publication
        pub_tmp = scientific.Publication(
            doi      = doi,
            citation = sci_dict[i]["citation"])

        # Append
        pub.append(pub_tmp)

    # Apply
    sci_ext.apply(publications = pub)

    # Return
    return collection