# Import
import os
from pathlib import Path
import numpy as np
import boto3
import math
from .utils.read import json_raw as read_json
from .utils.progress import bar as progress_bar
from multiprocessing.pool import ThreadPool

# upload
def upload(product_json_path,endpoint,credential,wcards):

    # Show progress
    print(" - STAC Uploading started")

    # Read collection's information
    product_dict = read_json(product_json_path)
    local_bucket = Path(endpoint["root_local"])\
                       .joinpath(endpoint["bucket"])
    key_root     = Path(endpoint["ftype"])\
                       .joinpath(endpoint["version_path"])\
                       .joinpath(product_dict["id"])

    # Parameter settings
    local = {"bucket"  : local_bucket,
             "key_root": key_root}
    cloud = {"bucket"  : endpoint["bucket"],
             "root"    : endpoint["root_cloud"]}

    # File detection
    filekeys_all = detect_all_files(local,wcards)

    # Decrease number
    #filekeys_all = filekeys_all[155000:]

    # Devide files with some number
    div_num = 200
    filekeys_div = np.array_split(
        np.array(filekeys_all),
        math.ceil(len(filekeys_all)/div_num)
    )

    # Show progress
    print(f" - Uploading {len(filekeys_all)} files ...")

    # Upload with divided numbers
    for i in range(len(filekeys_div)):

        # Initialization of files, keys, result index
        filekeys_tmp = filekeys_div[i]

        # Upload files until all files are uploaded
        while len(filekeys_tmp) > 0:

            # Upload (normal)
            #filekeys_tmp = upload_files_normal(filekeys_tmp,cloud,credential)
    
            # Upload (parallel)
            filekeys_tmp = upload_files_parallel(filekeys_tmp,cloud,credential)

            if len(filekeys_tmp) > 0:
                print("error happened")

        # Show progress
        progress_bar(i,len(filekeys_div))
    
    # Finish
    print(" - Uploading is finished")
    

# Upload files parallel
def upload_files_parallel(filekeys_all,cloud,credential):

    # Set RSTOR settings
    rstor = boto3.client(
        "s3",
        aws_access_key_id     = credential["RSTOR_ID"],
        aws_secret_access_key = credential["RSTOR_SECRET"],
        endpoint_url          = cloud["root"]
    )

    # Define upload function
    def upload(filekey):

        try:
            # Upload
            rstor.upload_file(
                Filename = str(filekey[0]),
                Bucket   = cloud["bucket"],
                Key      = filekey[1].as_posix()
            )
            return None

        except:

            # Return error file,key
            return filekey

    # Set thread pool (processes = os.cpu_count()-1)
    pool = ThreadPool(processes = os.cpu_count()-1)

    # Upload by starmap
    result = pool.map(upload,filekeys_all)

    # Close pool
    pool.close()

    # Delete None from result
    filekeys_err = list(filter(None,result))

    # Return
    return filekeys_err

# Upload files normal
def upload_files_normal(filekeys_all,cloud,credential): 

    # Set RSTOR settings
    rstor = boto3.client(
        "s3",
        aws_access_key_id     = credential["RSTOR_ID"],
        aws_secret_access_key = credential["RSTOR_SECRET"],
        endpoint_url          = cloud["root"]
    )

    # Initilization
    filekeys_err = []

    # Upload object
    for i in range(len(filekeys_all)):

        try:

            # Upload
            rstor.upload_file(
                Filename = str(filekeys_all[i][0]),
                Bucket   = cloud["bucket"],
                Key      = filekeys_all[i][1].as_posix()
            )

        except:

            # Record error index
            filekeys_err.append(filekeys_all[i])

    # Return
    return filekeys_err

# detect_all_files
def detect_all_files(local,wcards):

    # Show progress
    print(" - Detecting all files ... ")

    # Set folder path
    folder_path = local["bucket"].joinpath(local["key_root"])

    # Detect all subdirectories
    dir_all = np.array(list(Path(folder_path).glob("**")))

    # Search all subdirectories
    files_all = []
    keys_all  = []
    for i in range(len(dir_all)):
        
        # All files
        if len(wcards) == 0:

            # All subdirectories and files
            path_tmp = Path(dir_all[i]).iterdir()

            # Extract only files
            files_tmp = [p for p in path_tmp if p.is_file()]

        # Extract specific file types
        else:
            files_tmp = []
            for j in range(len(wcards)):
                ftmp = Path(dir_all[i]).glob(wcards[j])
                files_tmp.extend(ftmp)

        # Extract key names
        keys_tmp  = [p.relative_to(local["bucket"]) for p in files_tmp]

        # Extend
        files_all.extend(files_tmp)
        keys_all.extend(keys_tmp)

        # Show progress
        progress_bar(i,len(dir_all))

    # Merge two factor
    filekeys = list(zip(files_all,keys_all))

    # Return
    return filekeys
